import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, CheckCircle, Camera, Image, Film, Home, Zap, Award, TrendingUp, Clock, Building2, Star } from 'lucide-react'
import { motion } from 'framer-motion'
import './BookStudio.css'

const BookStudio = () => {
  const navigate = useNavigate()
  const [selectedType, setSelectedType] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const studioTypes = [
    { 
      id: 'backdrop', 
      name: 'Студия с фоном', 
      icon: Image,
      description: 'Профессиональные фоны и освещение'
    },
    { 
      id: 'chromakey', 
      name: 'Хромакей', 
      icon: Film,
      description: 'Зеленая стена для видеосъемки'
    },
    { 
      id: 'interior', 
      name: 'Интерьер', 
      icon: Home,
      description: 'Уютные интерьерные пространства'
    },
    { 
      id: 'openspace', 
      name: 'Open Space', 
      icon: Zap,
      description: 'Большое открытое пространство'
    }
  ]

  const studios = [
    { name: 'StudioSpace Premium', location: 'Центр', rating: 5, types: 'Все типы' },
    { name: 'StudioSpace Chroma', location: 'Тверская', rating: 5, types: 'Хромакей, Фон' },
    { name: 'StudioSpace Interior', location: 'Арбат', rating: 5, types: 'Интерьер, Open Space' },
    { name: 'StudioSpace Creative', location: 'Сокольники', rating: 5, types: 'Все типы' }
  ]

  const guarantees = [
    { icon: Award, title: 'Профессиональное оборудование', text: 'Лучшая техника' },
    { icon: Zap, title: 'Быстрое бронирование', text: 'За минуты' },
    { icon: Clock, title: 'Гибкий график', text: 'Работаем 24/7' },
    { icon: TrendingUp, title: 'Высокий рейтинг', text: '98% довольных клиентов' }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedType && selectedDate) {
      const savedBookings = localStorage.getItem('studioBookings')
      const bookings = savedBookings ? JSON.parse(savedBookings) : []
      
      const typeData = studioTypes.find(t => t.id === selectedType)
      const randomStudio = studios[Math.floor(Math.random() * studios.length)]
      
      const newBooking = {
        id: Date.now(),
        studio: randomStudio.name,
        date: selectedDate,
        equipment: typeData.description,
        review: ''
      }

      bookings.push(newBooking)
      localStorage.setItem('studioBookings', JSON.stringify(bookings))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedType('')
        setSelectedDate('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="book-studio-page">
      <div className="container">
        <div className="book-header">
          <motion.h1 
            className="book-title"
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.6 }}
          >
            Забронировать студию
          </motion.h1>
          <motion.p 
            className="book-subtitle"
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.6, delay: 0.1 }}
          >
            Выберите тип студии и дату
          </motion.p>
        </div>

        <motion.section 
          className="booking-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <form className="booking-form" onSubmit={handleSubmit}>
            <div className="form-group">
              <label className="form-label">
                <Camera size={18} strokeWidth={2} />
                Тип студии
              </label>
              <div className="types-list">
                {studioTypes.map((type) => {
                  const IconComponent = type.icon
                  return (
                    <motion.button
                      key={type.id}
                      type="button"
                      className={`type-item ${selectedType === type.id ? 'selected' : ''}`}
                      onClick={() => setSelectedType(type.id)}
                      whileHover={{ x: 3, transition: { duration: 0.2 } }}
                      whileTap={{ scale: 0.98 }}
                    >
                      <div className="type-item-icon">
                        <IconComponent size={24} strokeWidth={2} />
                      </div>
                      <div className="type-item-content">
                        <h3 className="type-item-name">{type.name}</h3>
                        <p className="type-item-desc">{type.description}</p>
                      </div>
                    </motion.button>
                  )
                })}
              </div>
            </div>

            <div className="form-group">
              <label htmlFor="date" className="form-label">
                <Calendar size={18} strokeWidth={2} />
                Дата бронирования
              </label>
              <input
                type="date"
                id="date"
                className="form-input"
                value={selectedDate}
                onChange={(e) => setSelectedDate(e.target.value)}
                min={minDate}
                max={maxDate}
                required
              />
            </div>

            <motion.button 
              type="submit" 
              className="submit-button" 
              disabled={!selectedType || !selectedDate}
              whileHover={{ y: -2 }}
              whileTap={{ scale: 0.98 }}
            >
              Забронировать
            </motion.button>

            {isSubmitted && (
              <motion.div 
                className="success-message"
                initial={{ opacity: 0, y: 10 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ duration: 0.3 }}
              >
                <CheckCircle size={20} strokeWidth={2} />
                <span>Вы успешно забронировали студию!</span>
              </motion.div>
            )}
          </form>
        </motion.section>

        <motion.section 
          className="guarantees-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <h2 className="section-title-minimal">Гарантии</h2>
          <div className="guarantees-grid">
            {guarantees.map((guarantee, index) => {
              const IconComponent = guarantee.icon
              return (
                <motion.div
                  key={guarantee.title}
                  className="guarantee-item"
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: index * 0.1, duration: 0.5 }}
                  whileHover={{ y: -3, transition: { duration: 0.2 } }}
                >
                  <div className="guarantee-icon">
                    <IconComponent size={24} strokeWidth={2} />
                  </div>
                  <div className="guarantee-content">
                    <h3 className="guarantee-title">{guarantee.title}</h3>
                    <p className="guarantee-text">{guarantee.text}</p>
                  </div>
                </motion.div>
              )
            })}
          </div>
        </motion.section>

        <motion.section 
          className="studios-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.5 }}
        >
          <h2 className="section-title-minimal">Наши студии</h2>
          <div className="studios-list">
            {studios.map((studio, index) => (
              <motion.div
                key={studio.name}
                className="studio-item"
                initial={{ opacity: 0, scale: 0.9 }}
                animate={{ opacity: 1, scale: 1 }}
                transition={{ delay: index * 0.1, duration: 0.5 }}
                whileHover={{ y: -3, transition: { duration: 0.2 } }}
              >
                <div className="studio-item-left">
                  <div className="studio-item-icon">
                    <Building2 size={24} strokeWidth={2} />
                  </div>
                  <div className="studio-item-content">
                    <h3 className="studio-item-name">{studio.name}</h3>
                    <div className="studio-item-details">
                      <span className="studio-item-location">{studio.location}</span>
                      <div className="studio-item-rating">
                        {Array.from({ length: 5 }, (_, i) => (
                          <Star key={i} size={14} fill="#F59E0B" stroke="#F59E0B" strokeWidth={2} />
                        ))}
                      </div>
                    </div>
                    <p className="studio-item-types">{studio.types}</p>
                  </div>
                </div>
              </motion.div>
            ))}
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default BookStudio

